<?php

class Byteonic_Admin
{
    private $api;
    private $message = '';
    private $message_type = '';

    public function __construct($api)
    {
        $this->api = $api;
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'handle_actions']);
        add_filter('plugin_action_links_' . plugin_basename(BYTEONIC_INTAKE_PATH . 'intake-forms.php'), [$this, 'add_settings_link']);
    }

    public function add_settings_link($links)
    {
        $settings_link = '<a href="admin.php?page=byteonic-intake">' . __('Settings', 'byteonic-intake') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    public function add_admin_menu()
    {
        add_menu_page(
            __('Byteonic Intake', 'byteonic-intake'),
            __('Byteonic Intake', 'byteonic-intake'),
            'manage_options',
            'byteonic-intake',
            [$this, 'render_settings_page'],
            'dashicons-cloud',
            56
        );
    }

    public function handle_actions()
    {
        if (!isset($_POST['byteonic_action']) || !current_user_can('manage_options')) {
            return;
        }

        if (!check_admin_referer('byteonic_save_settings', 'byteonic_nonce')) {
            return;
        }

        if ($_POST['byteonic_action'] === 'save_key') {
            $key = sanitize_text_field($_POST['api_key']);

            // Test connection before saving
            $test = $this->api->test_connection($key);

            if (is_wp_error($test)) {
                $this->message = __('Connection failed: ', 'byteonic-intake') . $test->get_error_message();
                $this->message_type = 'error';
            } else {
                $this->api->set_api_key($key);
                $this->message = __('Connected successfully! Settings saved.', 'byteonic-intake');
                $this->message_type = 'success';
            }
        } elseif ($_POST['byteonic_action'] === 'disconnect') {
            $this->api->delete_api_key();
            $this->message = __('Disconnected from Byteonic Intake.', 'byteonic-intake');
            $this->message_type = 'success';
        }
    }

    public function render_settings_page()
    {
        $api_key = $this->api->get_api_key();
        $is_connected = !empty($api_key);
        $forms = $is_connected ? $this->api->get_forms() : [];
        $forms_count = is_wp_error($forms) ? 0 : count($forms);
        $forms_error = is_wp_error($forms) ? $forms->get_error_message() : '';

        ?>
        <div class="wrap">
            <h1>
                <?php _e('Byteonic Intake Settings', 'byteonic-intake'); ?>
            </h1>

            <?php if ($this->message): ?>
                <div class="notice notice-<?php echo esc_attr($this->message_type); ?> is-dismissible">
                    <p>
                        <?php echo esc_html($this->message); ?>
                    </p>
                </div>
            <?php endif; ?>

            <div class="card" style="max-width: 600px; margin-top: 20px; padding: 20px;">
                <h2>
                    <?php _e('API Configuration', 'byteonic-intake'); ?>
                </h2>
                <p>
                    <?php _e('Enter your Byteonic Intake API Key to connect your forms.', 'byteonic-intake'); ?>
                </p>

                <form method="post" action="">
                    <?php wp_nonce_field('byteonic_save_settings', 'byteonic_nonce'); ?>

                    <?php if ($is_connected): ?>
                        <div style="background: #f0f9eb; padding: 10px; border-left: 4px solid #46b450; margin-bottom: 20px;">
                            <p><strong>
                                    <?php _e('Status: Connected', 'byteonic-intake'); ?>
                                </strong></p>
                            <p>
                                <?php _e('API Key:', 'byteonic-intake'); ?>
                                <?php echo esc_html(substr($api_key, 0, 8) . '...' . substr($api_key, -4)); ?>
                            </p>
                        </div>

                        <input type="hidden" name="byteonic_action" value="disconnect">
                        <p>
                            <button type="submit" class="button button-secondary"
                                onclick="return confirm('Are you sure you want to disconnect?');">
                                <?php _e('Disconnect', 'byteonic-intake'); ?>
                            </button>
                        </p>

                    <?php else: ?>

                        <table class="form-table">
                            <tr>
                                <th scope="row"><label for="api_key">
                                        <?php _e('API Key', 'byteonic-intake'); ?>
                                    </label></th>
                                <td>
                                    <input type="text" name="api_key" id="api_key" class="regular-text" value=""
                                        placeholder="byt_live_..." required>
                                    <p class="description">
                                        <?php _e('You can generate an API key in your Byteonic Intake dashboard under Settings.', 'byteonic-intake'); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>

                        <input type="hidden" name="byteonic_action" value="save_key">
                        <p class="submit">
                            <button type="submit" class="button button-primary">
                                <?php _e('Connect', 'byteonic-intake'); ?>
                            </button>
                        </p>

                    <?php endif; ?>
                </form>
            </div>

            <?php if ($is_connected): ?>
                <div class="card" style="max-width: 600px; margin-top: 20px; padding: 20px;">
                    <h2>
                        <?php _e('Available Forms', 'byteonic-intake'); ?>
                    </h2>
                    <?php if ($forms_error): ?>
                        <div class="notice notice-error inline">
                            <p>
                                <?php echo esc_html($forms_error); ?>
                            </p>
                        </div>
                    <?php elseif ($forms_count > 0): ?>
                        <p>
                            <?php printf(__('Found %d forms in your organization:', 'byteonic-intake'), $forms_count); ?>
                        </p>
                        <ul style="list-style: disc; margin-left: 20px;">
                            <?php foreach ($forms as $form): ?>
                                <li>
                                    <strong>
                                        <?php echo esc_html($form['name']); ?>
                                    </strong>
                                    <code>(<?php echo esc_html($form['slug']); ?>)</code>
                                    -
                                    <?php echo intval($form['fields_count']); ?> fields
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    <?php else: ?>
                        <p>
                            <?php _e('No active forms found in your organization.', 'byteonic-intake'); ?>
                        </p>
                    <?php endif; ?>
                </div>

                <div class="card" style="max-width: 600px; margin-top: 20px; padding: 20px;">
                    <h2>
                        <?php _e('Integrations', 'byteonic-intake'); ?>
                    </h2>
                    <p>
                        <?php _e('To connect a form:', 'byteonic-intake'); ?>
                    </p>
                    <ol>
                        <li>
                            <?php _e('Go to your form plugin settings (Gravity Forms, CF7, etc.)', 'byteonic-intake'); ?>
                        </li>
                        <li>
                            <?php _e('Look for the "Byteonic Intake" settings TAB or Panel.', 'byteonic-intake'); ?>
                        </li>
                        <li>
                            <?php _e('Select one of the forms listed above and map your fields.', 'byteonic-intake'); ?>
                        </li>
                    </ol>

                    <h3>
                        <?php _e('Detected Plugins', 'byteonic-intake'); ?>
                    </h3>
                    <ul>
                        <li>
                            Gravity Forms:
                            <?php echo class_exists('GFForms') ? '<span style="color:green">Active</span>' : '<span style="color:red">Not Active</span>'; ?>
                        </li>
                        <li>
                            Contact Form 7:
                            <?php echo defined('WPCF7_VERSION') ? '<span style="color:green">Active</span>' : '<span style="color:red">Not Active</span>'; ?>
                        </li>
                        <li>
                            WPForms:
                            <?php echo class_exists('WPForms') ? '<span style="color:green">Active</span>' : '<span style="color:red">Not Active</span>'; ?>
                        </li>
                        <li>
                            Fluent Forms:
                            <?php echo (defined('FLUENTFORM') || class_exists('FluentForm\Framework\Foundation\Application')) ? '<span style="color:green">Active</span>' : '<span style="color:red">Not Active</span>'; ?>
                        </li>
                    </ul>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}
