<?php
class Byteonic_CF7 {
    private $api;

    public function __construct( $api ) {
        $this->api = $api;

        if ( defined( 'WPCF7_VERSION' ) ) {
            add_filter( 'wpcf7_editor_panels', [ $this, 'add_editor_panel' ] );
            add_action( 'wpcf7_save_contact_form', [ $this, 'save_form_settings' ] );
            add_action( 'wpcf7_before_send_mail', [ $this, 'handle_submission' ] );
        }
    }

    public function add_editor_panel( $panels ) {
        $panels['byteonic-intake'] = [
            'title' => __( 'Byteonic Intake', 'byteonic-intake' ),
            'callback' => [ $this, 'render_editor_panel' ],
        ];
        return $panels;
    }

    public function render_editor_panel( $post ) {
        $settings = get_post_meta( $post->id(), 'byteonic_intake_settings', true );
        $enabled = isset( $settings['enabled'] ) ? $settings['enabled'] : false;
        $saved_slug = isset( $settings['form_slug'] ) ? $settings['form_slug'] : '';
        $field_map = isset( $settings['field_map'] ) ? $settings['field_map'] : [];
        
        $available_forms = $this->api->get_forms();
        
        // CF7 Tags
        $tags = $post->scan_form_tags();
        $cf7_fields = [];
        foreach ( $tags as $tag ) {
            if ( ! empty( $tag->name ) ) {
                $cf7_fields[] = $tag->name;
            }
        }
        ?>
        <div style="padding: 20px; background: #f9f9f9; border: 1px solid #e5e5e5; border-radius: 5px;">
            <div style="display: flex; align-items: center; margin-bottom: 20px;">
                 <!-- Branding could go here if image assets available -->
                <h2 style="margin:0; color: #4F46E5;"><?php _e( 'Byteonic Intake Integration', 'byteonic-intake' ); ?></h2>
                <span style="margin-left: 10px; font-size: 12px; background: #e0e7ff; color: #3730a3; padding: 2px 8px; border-radius: 12px; border: 1px solid #c7d2fe;">v<?php echo BYTEONIC_INTAKE_VERSION; ?></span>
            </div>
            <br>
            
            <label>
                <input type="checkbox" name="byteonic_enabled" value="1" <?php checked( $enabled, true ); ?>>
                <?php _e( 'Enable Integration', 'byteonic-intake' ); ?>
            </label>
            
            <br><br>
            
            <label><?php _e( 'Select Target Form:', 'byteonic-intake' ); ?></label><br>
            <?php if ( is_wp_error( $available_forms ) ): ?>
                <div style="color:red;"><?php echo $available_forms->get_error_message(); ?></div>
            <?php else: ?>
                <select name="byteonic_form_slug" style="min-width: 300px;">
                    <option value=""><?php _e( 'Select a form...', 'byteonic-intake' ); ?></option>
                    <?php foreach ( $available_forms as $form ): ?>
                        <option value="<?php echo esc_attr( $form['slug'] ); ?>" <?php selected( $saved_slug, $form['slug'] ); ?>>
                            <?php echo esc_html( $form['name'] ); ?> (<?php echo esc_html( $form['slug'] ); ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            <?php endif; ?>

            <?php if ( $saved_slug && ! is_wp_error( $available_forms ) ):
                 $target_form = $this->api->get_form( $saved_slug );
                 if ( $target_form ):
                    if ( ! empty( $target_form['fields'] ) ):
            ?>
                <br><br>
                <h3><?php _e( 'Field Mapping', 'byteonic-intake' ); ?></h3>
                <p>Map your Contact Form 7 fields (Mail Tags) to Byteonic fields.</p>
                <table class="widefat fixed" style="width: auto; max-width: 600px;">
                    <thead>
                        <tr>
                            <th><?php _e( 'Byteonic Field', 'byteonic-intake' ); ?></th>
                            <th><?php _e( 'CF7 Mail Tag', 'byteonic-intake' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $target_form['fields'] as $b_field ): ?>
                            <tr>
                                <td>
                                    <strong><?php echo esc_html( $b_field['label'] ); ?></strong>
                                    <?php if ( ! empty( $b_field['required'] ) ) echo ' <span style="color:red">*</span>'; ?>
                                </td>
                                <td>
                                    <select name="byteonic_field_map[<?php echo esc_attr( $b_field['key'] ); ?>]">
                                        <option value=""><?php _e( 'Select tag...', 'byteonic-intake' ); ?></option>
                                        <?php foreach ( $cf7_fields as $tag_name ):
                                            $map_value = isset( $field_map[ $b_field['key'] ] ) ? $field_map[ $b_field['key'] ] : '';
                                        ?>
                                            <option value="<?php echo esc_attr( $tag_name ); ?>" <?php selected( $map_value, $tag_name ); ?>>
                                                [<?php echo esc_html( $tag_name ); ?>]
                                            </option>
                                        <?php endforeach; ?>
                                        <option value="custom_value" <?php selected( $map_value, 'custom_value' ); ?>>-- Custom Value --</option>
                                    </select>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                 <?php else: ?>
                    <br><br>
                    <div style="background: #fff3cd; padding: 10px; border-left: 4px solid #ffc107;">
                        <p><strong>Warning:</strong> The selected form ("<?php echo esc_html($target_form['name']); ?>") has no fields defined in Byteonic. Please add fields in your Byteonic Dashboard to enable mapping.</p>
                    </div>
                 <?php endif; ?>
                 <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

    public function save_form_settings( $contact_form ) {
        // CF7 doesn't pass $_POST directly to save action in a clean way for addons sometimes,
        // but we can access $_POST global as it is a form submission.
        
        $post_id = $contact_form->id();
        
        // Check permissions/nonce? CF7 admin handles this generally.
        
        if ( isset( $_POST['byteonic_form_slug'] ) ) {
             $settings = [
                'enabled' => isset( $_POST['byteonic_enabled'] ),
                'form_slug' => sanitize_text_field( $_POST['byteonic_form_slug'] ),
                'field_map' => isset( $_POST['byteonic_field_map'] ) ? array_map( 'sanitize_text_field', $_POST['byteonic_field_map'] ) : []
            ];
            update_post_meta( $post_id, 'byteonic_intake_settings', $settings );
        }
    }
    
    /**
     * Helper log function
     */
    private function log($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Byteonic Intake CF7] ' . $message);
        }
    }

    public function handle_submission( $contact_form ) {
        $submission = WPCF7_Submission::get_instance();
        if ( ! $submission ) return;
        
        $posted_data = $submission->get_posted_data();
        $post_id = $contact_form->id();
        $settings = get_post_meta( $post_id, 'byteonic_intake_settings', true );
        
        // Log entry
        $this->log("Processing submission for Form ID: $post_id");

        if ( empty( $settings ) || empty( $settings['enabled'] ) || empty( $settings['form_slug'] ) ) {
            $this->log("Integration disabled or invalid settings for Form ID: $post_id");
            return;
        }

        $slug = $settings['form_slug'];
        $field_map = isset( $settings['field_map'] ) ? $settings['field_map'] : [];
        $data = [];

        $this->log("Target Slug: $slug");

        foreach ( $field_map as $byteonic_key => $cf7_tag ) {
            // Handle Custom Values
            if ( $cf7_tag === 'custom_value' ) {
                // Future: support defined custom value input? For now skip or empty.
                continue;
            }

            if ( ! empty( $cf7_tag ) && isset( $posted_data[ $cf7_tag ] ) ) {
                 // CF7 data can be array (checkboxes), convert to string
                if ( is_array( $posted_data[ $cf7_tag ] ) ) {
                    $val = implode( ', ', $posted_data[ $cf7_tag ] );
                } else {
                    $val = $posted_data[ $cf7_tag ];
                }
                $data[ $byteonic_key ] = $val;
            } else {
                // Log missing field mapping attempt
                $this->log("Field mapping missed for Byteonic Key: $byteonic_key (Tag: $cf7_tag). Data not found in submission.");
            }
        }
        
        if (empty($data)) {
            $this->log("Warning: No data mapped. Sending empty payload.");
        }
        
        // Add Meta
        $meta       = $submission->get_meta( 'container_post_id' );
        $source_url = $meta ? get_permalink( $meta ) : '';
        // If empty, try unit tag or referer? wp_get_referer() is often reliable for AJAX submissions too.
        if ( empty( $source_url ) ) {
            $source_url = wp_get_referer();
        }

        $data['_meta'] = [
            'current_url' => $source_url,
            'remote_ip'   => $submission->get_meta( 'remote_ip' ),
            'user_agent'  => $submission->get_meta( 'user_agent' ),
            'plugin'      => 'cf7',
            'version'     => BYTEONIC_INTAKE_VERSION
        ];

        $this->log("Payload prepared: " . print_r($data, true));
        
        // Submit
        $result = $this->api->submit_form( $slug, $data );

        if ( is_wp_error( $result ) ) {
             $this->log("Submission Error: " . $result->get_error_message());
        } else {
             $this->log("Submission Success: " . print_r($result, true));
        }
    }
}
