<?php

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Byteonic_WPForms
 * Handles integration with WPForms
 */
class Byteonic_WPForms
{
    /**
     * @var Byteonic_API
     */
    private $api;

    public function __construct($api)
    {
        $this->api = $api;

        // Register even if WPForms class doesn't exist yet (late instantiation)
        // Sections Filter
        add_filter('wpforms_builder_settings_sections', [$this, 'add_settings_section'], 20, 2);

        // Content Action: wpforms_form_settings_panel_content
        // This is the correct hook found in WPForms 1.6.7+
        add_action('wpforms_form_settings_panel_content', [$this, 'render_settings_content'], 20, 1);

        add_action('wpforms_process_complete', [$this, 'handle_submission'], 10, 4);
    }

    public function add_settings_section($sections, $form_data)
    {
        // Slug: byteonic-intake (DASHES)
        $sections['byteonic-intake'] = __('Byteonic Intake', 'byteonic-intake');
        return $sections;
    }

    public function render_settings_content($instance)
    {
        // Critical Wrapper ID: wpforms-panel-content-section-byteonic-intake
        echo '<div class="wpforms-panel-content-section wpforms-panel-content-section-byteonic-intake" id="wpforms-panel-content-section-byteonic-intake">';

        // Title
        echo '<div class="wpforms-panel-content-section-title">';
        echo 'Byteonic Intake';
        echo '</div>';

        // Fetch form data
        $form_data = $instance->form_data;

        // Fetch available forms from API
        $forms = $this->api->get_forms();
        $form_options = [];

        if (is_wp_error($forms)) {
            echo '<div class="error inline"><p>' . esc_html($forms->get_error_message()) . '</p></div>';
        } else {
            foreach ($forms as $form) {
                $form_options[$form['slug']] = $form['name'] . ' (' . $form['slug'] . ')';
            }
        }

        // --- UI FIELDS ---

        // 1. Enable Toggle
        wpforms_panel_field(
            'toggle',
            'settings',
            'byteonic_enabled',
            $form_data,
            __('Enable Integration', 'byteonic-intake'),
            [
                'tooltip' => __('Enable to send form submissions to Byteonic Intake.', 'byteonic-intake')
            ]
        );

        // 2. Form Selection
        if (!empty($form_options)) {
            wpforms_panel_field(
                'select',
                'settings',
                'byteonic_form_slug',
                $form_data,
                __('Select Target Form', 'byteonic-intake'),
                [
                    'options' => $form_options,
                    'tooltip' => __('Select the Byteonic Intake form to send data to.', 'byteonic-intake'),
                    'class' => 'byteonic-form-slug-select', // Add class for potential JS listening
                    'data' => [ // Try to trigger save/refresh on change if possible, effectively requires save
                        'refresh' => 1
                    ]
                ]
            );
        } else {
            echo '<p>' . __('No forms found. Please check your API key or create a form in Byteonic Intake.', 'byteonic-intake') . '</p>';
        }

        // 3. Field Mapping
        $saved_slug = !empty($form_data['settings']['byteonic_form_slug']) ? $form_data['settings']['byteonic_form_slug'] : '';

        if ($saved_slug && !is_wp_error($forms)) {
            $target_form = $this->api->get_form($saved_slug);

            if ($target_form && !empty($target_form['fields'])) {
                echo '<div class="wpforms-panel-content-section-title">';
                echo __('Field Mapping', 'byteonic-intake');
                echo '</div>';
                echo '<p>' . __('Map your WPForms fields to Byteonic fields below.', 'byteonic-intake') . '</p>';

                // Get WPForms fields
                $wpforms_fields = !empty($form_data['fields']) ? $form_data['fields'] : [];
                $wpforms_options = [];
                $wpforms_options[''] = __('Select Field...', 'byteonic-intake');

                if (!empty($wpforms_fields)) {
                    foreach ($wpforms_fields as $field) {
                        $label = !empty($field['label']) ? $field['label'] : 'Field #' . $field['id'];
                        $wpforms_options[$field['id']] = $label;
                    }
                }

                // Render Table
                echo '<table class="wpforms-table-settings-list" style="width:100%; max-width:100%; border-collapse: collapse; margin-top: 10px;">';
                echo '<thead>
                        <tr style="border-bottom: 1px solid #ddd; text-align: left;">
                            <th style="padding: 10px;">' . __('Byteonic Field', 'byteonic-intake') . '</th>
                            <th style="padding: 10px;">' . __('WPForms Field', 'byteonic-intake') . '</th>
                        </tr>
                      </thead>';
                echo '<tbody>';

                foreach ($target_form['fields'] as $b_field) {
                    $b_key = $b_field['key'];
                    $b_label = $b_field['label'];
                    $required = !empty($b_field['required']) ? '<span class="required" style="color:red;">*</span>' : '';

                    // Saved Value
                    $saved_map = isset($form_data['settings']['byteonic_field_map'][$b_key]) ? $form_data['settings']['byteonic_field_map'][$b_key] : '';

                    echo '<tr style="border-bottom: 1px solid #eee;">';
                    echo '<td style="padding: 10px; vertical-align: middle;">
                            <strong>' . esc_html($b_label) . '</strong> ' . $required . '<br>
                            <small style="color: #666;">' . esc_html($b_key) . '</small>
                          </td>';
                    echo '<td style="padding: 10px; vertical-align: middle;">';

                    echo '<select name="settings[byteonic_field_map][' . esc_attr($b_key) . ']" style="width: 100%; max-width: 300px;">';
                    foreach ($wpforms_options as $wp_id => $wp_label) {
                        echo '<option value="' . esc_attr($wp_id) . '" ' . selected($saved_map, $wp_id, false) . '>' . esc_html($wp_label) . '</option>';
                    }
                    echo '</select>';

                    echo '</td>';
                    echo '</tr>';
                }

                echo '</tbody></table>';

            } elseif ($target_form) {
                echo '<p>' . __('The selected form has no fields to map.', 'byteonic-intake') . '</p>';
            }
        } elseif (!$saved_slug && !empty($form_options)) {
            echo '<p><em>' . __('Please select a form and save settings to configure mapping.', 'byteonic-intake') . '</em></p>';
        }

        echo '</div>'; // End section wrapper
    }

    /**
     * Helper log function
     */
    private function log($message)
    {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[Byteonic Intake WPForms] ' . $message);
        }
    }

    public function handle_submission($fields, $entry, $form_data, $entry_id)
    {
        // Check if enabled
        if (empty($form_data['settings']['byteonic_enabled'])) {
            return;
        }

        // Check if slug selected
        $slug = !empty($form_data['settings']['byteonic_form_slug']) ? $form_data['settings']['byteonic_form_slug'] : '';
        if (empty($slug)) {
            return;
        }

        $field_map = isset($form_data['settings']['byteonic_field_map']) ? $form_data['settings']['byteonic_field_map'] : [];
        $data = [];

        // Iterate through map
        foreach ($field_map as $b_key => $wp_field_id) {
            if (empty($wp_field_id))
                continue;

            if (isset($fields[$wp_field_id])) {
                $data[$b_key] = $fields[$wp_field_id]['value'];
            }
        }

        // Fallback: If no map, try auto-mapping (Legacy/Simple support)
        if (empty($field_map) && empty($data)) {
            foreach ($fields as $field) {
                // Map using Field Label as Key (heuristic) OR Field ID if needed
                $key = sanitize_title($field['name']);
                if (empty($key)) {
                    $key = 'field_' . $field['id'];
                }
                $data[$key] = $field['value'];
            }
        }

        // Check if data is empty
        if (empty($data)) {
            return;
        }

        // Add Meta
        $referer = wp_get_referer();
        $source_url = $referer ? $referer : home_url();

        $data['_meta'] = [
            'platform' => 'wordpress',
            'plugin' => 'wpforms',
            'form_id' => $form_data['id'],
            'entry_id' => $entry_id,
            'current_url' => $source_url
        ];

        // Send to API
        $this->api->submit_form($slug, $data);
    }
}

